using System;
using System.Data;
using System.Data.SqlTypes;
using System.Data.SqlClient;
using MethodBase = System.Reflection.MethodBase;
using gov.va.med.vbecs.ExceptionManagement;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using gov.va.med.vbecs.Common;

namespace gov.va.med.vbecs.DAL
{

		#region Header

		///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/9/2003</CreationDate>
		///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		///<summary></summary>

		#endregion

	public class OrderedTest
	{

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/9/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2607"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2608"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a boolean telling if the ordered test is pending
		/// </summary>
		/// <returns></returns>
		public static bool IsOrderedTestPending()
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.IsOrderedTestPending.divisionCode, System.Data.SqlDbType.VarChar)
				};

			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;

			return new StoredProcedure().GetBoolReturnValue( STOREDPROC.IsOrderedTestPending.StoredProcName, prms );
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/25/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2695"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>The datatable containing list of orderable tests</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1291"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the list of all types of tests
		/// </summary>
		/// <returns></returns>
		public static DataTable GetOrderableTestList()
		{
			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetOrderableTestList.StoredProcName);
			
			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3010"> 
		///		<ExpectedInput>Valid Orderable Test ID</ExpectedInput>
		///		<ExpectedOutput>A data table containing the list of orderable tests</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3011"> 
		///		<ExpectedInput>Invalid OrderableTestId</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the list of all reorderable tests based on the Parent Order Test ID
		/// </summary>
		/// <param name="orderableTestId"></param>
		/// <returns>datatable</returns>
		public static DataTable GetReorderableTestList(int orderableTestId)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetReorderableTestList.orderabletestid, System.Data.SqlDbType.Int)
			};

			if (orderableTestId == int.MinValue)
			{
				prms[0].Value = null;
			}
			else
			{
				prms[0].Value = orderableTestId;
			}

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetReorderableTestList.StoredProcName, prms);
			
			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1292"> 
		///		<ExpectedInput>OrderedTest Guid</ExpectedInput>
		///		<ExpectedOutput>DataTable of ordered test info</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1293"> 
		///		<ExpectedInput>Invalid OrderedTest Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetOrderedTest
		/// </summary>
		/// <param name="orderedTestGuid"></param>
		/// <returns></returns>
		public static DataTable GetOrderedTest(Guid orderedTestGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetOrderedTest.orderedtestguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetOrderedTest.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = orderedTestGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetOrderedTest.StoredProcName, prms);
			
			return ds.Tables[0];
		}
		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4026"> 
		///		<ExpectedInput>A valid dataset with four tables</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4027"> 
		///		<ExpectedInput>An invalid dataset with no tables</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Save Order Reflex tests data from UC_45. Saves data to up to four tables
		/// </summary>
		/// <param name="dsSave">DataSet with the following tables: dtPatientSpecimenComment, dtPatientOrder, dtOrderedTests, dtOrderedUnit</param>
		/// <returns>bool</returns>
		public static bool SaveReflexTests(DataSet dsSave)
		{
			int NumberOfTableToSave = dsSave.Tables.Count;
			int Counter = 0;
			int returnValue = 0;
			StoredProcedure sproc = new StoredProcedure();

			DataTable[] dtArray = new DataTable[NumberOfTableToSave];
			string[] spArray = new string[NumberOfTableToSave];

			foreach (DataTable dtSave in dsSave.Tables)
			{
				dtArray[Counter] = dtSave;

				if (dtSave.TableName == TABLE.PatientOrder.TableName)
				{
					spArray[Counter] = STOREDPROC.InsertPatientOrder.StoredProcName;
				}
				else if (dtSave.TableName == TABLE.OrderedTest.TableName)
				{
					spArray[Counter] = STOREDPROC.InsertOrderedTest.StoredProcName;
				}
				else if (dtSave.TableName == TABLE.OrderedUnit.TableName)
				{
					spArray[Counter] = STOREDPROC.InsertOrderedUnit.StoredProcName;
				}
				else if (dtSave.TableName == TABLE.ExceptionReflexTest.TableName)
				{
					spArray[Counter] = STOREDPROC.InsertExceptionReflexTest.StoredProcName;
				}
				Counter++;
			}

			returnValue = sproc.TransactionalGetValue(spArray, dtArray);

			return(returnValue == 0);
		}


		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/25/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2696"> 
		///		<ExpectedInput>Empty String</ExpectedInput>
		///		<ExpectedOutput>The datatable containing testing tasks list</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2697"> 
		///		<ExpectedInput>Bad groupnumber</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the list of pending testing tasks
		/// </summary>
		/// <returns></returns>
		public static DataTable GetTestingTaskList(string orderGroupNumber)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetTestingTaskList.divisioncode, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.GetTestingTaskList.ordergroupnumber, System.Data.SqlDbType.VarChar)
			};
			
			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;
			if (orderGroupNumber != "")
				prms[1].Value = orderGroupNumber;
			else
				prms[1].Value = DBNull.Value;


			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetTestingTaskList.StoredProcName, prms);
			
			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/25/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2698"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>The datatable containing the list of task statuses</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1294"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the list of all task statuses
		/// </summary>
		/// <returns></returns>
		public static DataTable GetTaskStatusList()
		{
			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetTaskStatusList.StoredProcName);
			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/9/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2803"> 
		///		<ExpectedInput>Valid patient specimen GUID</ExpectedInput>
		///		<ExpectedOutput>Data table with records</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2804"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Given a patient specimen GUID, gets a list of ordered tests
		/// </summary>
		/// <param name="patientSpecimenGuid"></param>
		/// <returns></returns>
		public static DataTable GetOrderedTestsforSpecimen(Guid patientSpecimenGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetOrderedTestsforSpecimen.patientspecimenguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetOrderedTestsforSpecimen.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = patientSpecimenGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode; 

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetOrderedTestsforSpecimen.StoredProcName, prms);
			return ds.Tables[0];
		}


		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/5/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2854"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable containing all test orders for the respective division</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2855"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a data table of all the tpatient test orders for a division
		/// </summary>
		/// <returns></returns>
		public static DataTable GetTestOrders()
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetPatientTestOrdersAll.divisioncode, System.Data.SqlDbType.VarChar)
			};
			
			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;			
			
			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetPatientTestOrdersAll.StoredProcName, prms);
			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/16/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3036"> 
		///		<ExpectedInput>Valid OrderedTest identifier</ExpectedInput>
		///		<ExpectedOutput>DataTable containing test result for the task</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3037"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///	Get previously saved test results for task 
		/// </summary>
		/// <param name="orderedTestGuid"></param>
		/// <returns></returns>
		public static DataTable GetTestResultsForTask(Guid orderedTestGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetTestResultsForTask.orderedtestguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetTestResultsForTask.divisioncode, System.Data.SqlDbType.Char)

			};

			prms[0].Value = orderedTestGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;
			
			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetTestResultsForTask.StoredProcName, prms);

			return ds.Tables[0];
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8049"> 
		///		<ExpectedInput>Valid OrderedTestGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8050"> 
		///		<ExpectedInput>Invalid OrderedTestGuid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///	Get test results that were marked as errorneous for task 
		/// </summary>
		/// <param name="orderedTestGuid"></param>
		/// <returns></returns>
		public static DataTable GetTestResultsWithErrorForTask(Guid orderedTestGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetTestResultsWithErrorForTask.orderedtestguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetTestResultsWithErrorForTask.divisioncode, System.Data.SqlDbType.Char)

			};

			prms[0].Value = orderedTestGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;
			
			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetTestResultsWithErrorForTask.StoredProcName, prms);

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/16/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3038"> 
		///		<ExpectedInput>DataSet containing test results for tasks</ExpectedInput>
		///		<ExpectedOutput>True</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3039"> 
		///		<ExpectedInput>Invalid DataSet</ExpectedInput>
		///		<ExpectedOutput>False</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Save test result for tasks
		/// </summary>
		/// <param name="dsSave"></param>
		/// <returns></returns>
		public static bool SaveTestResults(DataSet dsSave)
		{
			int NumberOfTableToSave = dsSave.Tables.Count;

			if (NumberOfTableToSave == 0)
				return false;

			if (dsSave.Tables.IndexOf(TABLE.RackLot.TableName) != -1)
				NumberOfTableToSave++;

			int Counter = 0;

			DataTable[] dtArray = new DataTable[NumberOfTableToSave];
			string[] spArray = new string[NumberOfTableToSave];

			foreach (DataTable dtSave in dsSave.Tables)
			{

				if (dtSave.TableName == "SpecimenTestInsert")
				{
					if (dtSave.Rows.Count > 0)
					{
						dtArray[Counter] = dtSave;
						spArray[Counter] = STOREDPROC.InsertSpecimenTest.StoredProcName;
					}
				}
				else if (dtSave.TableName == "SpecimenTestUpdate")
				{
					if (dtSave.Rows.Count > 0)
					{
						dtArray[Counter] = dtSave;
						spArray[Counter] = STOREDPROC.UpdateSpecimenTestForPatientTesting.StoredProcName;
					}
				}
				else if (dtSave.TableName == "BloodUnitTestInsert")
				{
					if (dtSave.Rows.Count > 0)
					{
						dtArray[Counter] = dtSave;
						spArray[Counter] = STOREDPROC.InsertBloodUnitTest.StoredProcName;
					}

				}
				else if (dtSave.TableName == "BloodUnitTestUpdate")
				{
					if (dtSave.Rows.Count > 0)
					{
						dtArray[Counter] = dtSave;
						spArray[Counter] = STOREDPROC.UpdateBloodUnitTestResult.StoredProcName;
					}

				}
				else if (dtSave.TableName == TABLE.OrderedTest.TableName)
				{
					if (dtSave.Rows.Count > 0)
					{
						dtArray[Counter] = dtSave;
						spArray[Counter] = STOREDPROC.UpdateOrderedTestTaskStatus.StoredProcName;
					}
				}
				else if (dtSave.TableName == TABLE.RackLot.TableName)
				{
					if (dtSave.Rows.Count > 0)
					{
						DataTable dtUpdateRackLot = dtSave.Clone();
						DataTable dtInsertRackLot = dtSave.Clone();

						//splitting RackLot table into two - update and insert
						foreach(DataRow drRackLot in dtSave.Rows)
						{
							if (drRackLot.IsNull(TABLE.RackLot.RowVersion))
							{
								dtInsertRackLot.Rows.Add(drRackLot.ItemArray);
							}
							else
							{
								dtUpdateRackLot.Rows.Add(drRackLot.ItemArray);
							}
						}

						if (dtInsertRackLot.Columns.Contains(TABLE.RackLot.RowVersion))
						{
							dtInsertRackLot.Columns.Remove(TABLE.RackLot.RowVersion);
						}

						dtArray[Counter] = dtUpdateRackLot;
						spArray[Counter] = STOREDPROC.UpdateRackLot.StoredProcName;
						Counter++;
						dtArray[Counter] = dtInsertRackLot;
						spArray[Counter] = STOREDPROC.InsertRackLot.StoredProcName;
					}
				}
				else if (dtSave.TableName == "RackTestInsert")
				{
					if (dtSave.Rows.Count > 0)
					{
						dtArray[Counter] = dtSave;
						spArray[Counter] = STOREDPROC.InsertRackTest.StoredProcName;
					}
				}
				else if (dtSave.TableName == "RackTestUpdate")
				{
					if (dtSave.Rows.Count > 0)
					{
						dtArray[Counter] = dtSave;
						spArray[Counter] = STOREDPROC.UpdateRackTest.StoredProcName;
					}
				}
				else if (dtSave.TableName == TABLE.AntiseraOrderedTest.TableName)
				{
					if (dtSave.Rows.Count > 0)
					{
						dtArray[Counter] = dtSave;
						spArray[Counter] = STOREDPROC.InsertAntiseraOrderedTest.StoredProcName;
					}
				}
				else if (dtSave.TableName == TABLE.PatientTransfusionRequirement.TableName)
				{
					if (dtSave.Rows.Count > 0)
					{
						dtArray[Counter] = dtSave;
						spArray[Counter] = STOREDPROC.InsertTransfusionRequirements.StoredProcName;
					}
				}
				else if (dtSave.TableName == TABLE.OrderedUnit.TableName)
				{
					if (dtSave.Rows.Count > 0)
					{
						dtArray[Counter] = dtSave;
						spArray[Counter] = STOREDPROC.UpdateOrderedUnit.StoredProcName;
					}
				}
				else if (dtSave.TableName == TABLE.ExceptionQc.TableName)
				{
					if (dtSave.Rows.Count > 0)
					{
						dtArray[Counter] = dtSave;
						spArray[Counter] = STOREDPROC.InsertExceptionQC.StoredProcName;
					}
				}
				else if (dtSave.TableName == TABLE.ExceptionQcNotPerformed.TableName)
				{
					if (dtSave.Rows.Count > 0)
					{
						dtArray[Counter] = dtSave;
						spArray[Counter] = STOREDPROC.InsertExceptionQCNotPerformed.StoredProcName;
					}
				}
				else if (dtSave.TableName == TABLE.ExceptionPatientAboRhDiscrepancy.TableName)
				{
					if (dtSave.Rows.Count > 0)
					{
						dtArray[Counter] = dtSave;
						spArray[Counter] = STOREDPROC.InsertExceptionPatientAboRHDiscrepancy.StoredProcName;
					}
				}
				else if (dtSave.TableName == TABLE.ExceptionPatientAntigenDiscrepancy.TableName)
				{
					if (dtSave.Rows.Count > 0)
					{
						dtArray[Counter] = dtSave;
						spArray[Counter] = STOREDPROC.InsertExceptionPatientAntigenDiscrepancy.StoredProcName;
					}
				}
				else if (dtSave.TableName == TABLE.ExceptionPatientAboRh.TableName)
				{
					if (dtSave.Rows.Count > 0)
					{
						dtArray[Counter] = dtSave;
						spArray[Counter] = STOREDPROC.InsertExceptionPatientAboRH.StoredProcName;
					}
				}
				else if (dtSave.TableName == TABLE.ExceptionPatientTest.TableName)
				{
					if (dtSave.Rows.Count > 0)
					{
						dtArray[Counter] = dtSave;
						spArray[Counter] = STOREDPROC.InsertExceptionPatientTest.StoredProcName;
					}
				}
				else if (dtSave.TableName == TABLE.ExceptionUnitSelection.TableName)
				{
					if (dtSave.Rows.Count > 0)
					{
						dtArray[Counter] = dtSave;
						spArray[Counter] = STOREDPROC.InsertExceptionUnitSelection.StoredProcName;
					}
				}
				else if (dtSave.TableName == TABLE.ExceptionEmergencyIssueUnitTest.TableName)
				{
					if (dtSave.Rows.Count > 0)
					{
						dtArray[Counter] = dtSave;
						spArray[Counter] = STOREDPROC.InsertExceptionEmergencyIssueUnitTest.StoredProcName;
					}
				}
				else if (dtSave.TableName == TABLE.ExceptionExpiredTask.TableName)
				{
					if (dtSave.Rows.Count > 0)
					{
						dtArray[Counter] = dtSave;
						spArray[Counter] = STOREDPROC.InsertExceptionExpiredTask.StoredProcName;
					}
				}
				else if (dtSave.TableName == TABLE.WorkloadEvent.TableName)
				{
					if (dtSave.Rows.Count > 0)
					{
						dtArray[Counter] = dtSave;
						spArray[Counter] = STOREDPROC.InsertWorkloadEvents.StoredProcName;
					}
				}
				else if (dtSave.TableName == TABLE.ExceptionInconclusiveCrossmatch.TableName)
				{
					if (dtSave.Rows.Count > 0)
					{
						dtArray[Counter] = dtSave;
						spArray[Counter] = STOREDPROC.InsertExceptionInconclusiveCrossmatch.StoredProcName;
					}
				}
				//CR 2945
				else if (dtSave.TableName == TABLE.MessageLog.TableName)
				{
					if (dtSave.Rows.Count > 0)
					{
						dtArray[Counter] = dtSave;
						spArray[Counter] = STOREDPROC.HL7InsertMessageLog.StoredProcName;
					}
				}
				Counter++;
			}

			return ( (new StoredProcedure()).TransactionalGetValue( spArray, dtArray ) == 0 );
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/5/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2852"> 
		///		<ExpectedInput>Valid DataTables and parameters</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2853"> 
		///		<ExpectedInput>Invalid DataTables and parameters</ExpectedInput>
		///		<ExpectedOutput>RowVersion Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Updates the status of an ordered test
		/// </summary>
		/// <param name="orderedTestGuid"></param>
		/// <param name="orderStatusCode"></param>
		/// <param name="taskStatusCode"></param>
		/// <param name="userName"></param>
		/// <param name="cannedCommentGuid"></param>
		/// <param name="cannedCommentText"></param>
		/// <param name="cprsOrderNumber"></param>
		/// <param name="dtSpecimen"></param>
		/// <param name="dtOrderedTest"></param>
		/// <param name="dtWorkloadEvents"></param>
		/// <param name="lastUpdateFunctionId">Uc or calling method</param>
		/// <returns></returns>
		public static bool UpdateStatus(Guid orderedTestGuid, 
			char orderStatusCode, 
			char taskStatusCode, 
			string userName, 
			Guid cannedCommentGuid, 
			string cannedCommentText, 
			string cprsOrderNumber, 
			DataTable dtSpecimen, 
			DataTable dtOrderedTest, 
			DataTable dtWorkloadEvents,
			Common.UpdateFunction lastUpdateFunctionId)
		{	
			int returnCode;
			System.Collections.ArrayList _dtArrayList = new System.Collections.ArrayList();
			System.Collections.ArrayList _sprocArrayList = new System.Collections.ArrayList();
						
			dtOrderedTest = Common.Utility.AppendLastUpdateInformation(dtOrderedTest, lastUpdateFunctionId);
			if(dtWorkloadEvents != null)
			{
				dtWorkloadEvents = Common.Utility.AppendLastUpdateInformation(dtWorkloadEvents, lastUpdateFunctionId);
				_dtArrayList.Add(dtWorkloadEvents);
				_sprocArrayList.Add(STOREDPROC.InsertWorkloadEvents.StoredProcName);
			}
			if (dtSpecimen != null)
			{
				dtSpecimen = Common.Utility.AppendLastUpdateInformation(dtSpecimen, lastUpdateFunctionId);
				
				//Add table with specimen data
				_dtArrayList.Add(dtSpecimen.GetChanges(DataRowState.Added));
				_sprocArrayList.Add(STOREDPROC.InsertPatientSpecimen.StoredProcName);
			}

			_dtArrayList.Add(dtOrderedTest);
			_sprocArrayList.Add(Common.VbecsStoredProcs.UpdateOrderedTestStatus.StoredProcName);

			returnCode = (new StoredProcedure()).TransactionalGetValue(_sprocArrayList, _dtArrayList);
			
			//check the final status of return code
			return(returnCode == 0);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1300"> 
		///		<ExpectedInput>CprsOrderNumber</ExpectedInput>
		///		<ExpectedOutput>DataSet of Order Data</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1304"> 
		///		<ExpectedInput>Invalid CprsNumber</ExpectedInput>
		///		<ExpectedOutput>Empty DataSet</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets order data for CPRS update
		/// </summary>
		/// <param name="cprsOrderNumber"></param>
		/// <returns></returns>
		public static DataSet GetOrderData(string cprsOrderNumber)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetOrderDataTests.cprsordernumber, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.GetOrderDataTests.divisioncode, System.Data.SqlDbType.VarChar),
			};

			prms[0].Value = cprsOrderNumber;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			return Common.StoredProcedure.GetData(STOREDPROC.GetOrderDataTests.StoredProcName, prms);
		}


		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/16/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3040"> 
		///		<ExpectedInput>OrderedTest identifier</ExpectedInput>
		///		<ExpectedOutput>DataTable containing QC results associated with OrderedTest</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3041"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get QC results for task
		/// </summary>
		/// <param name="orderedTestGuid"></param>
		/// <returns></returns>
		public static DataTable GetQCResultsForOrderedTest(Guid orderedTestGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetQCResultsForOrderedTest.orderedtestguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetQCResultsForOrderedTest.divisioncode, System.Data.SqlDbType.Char)
			};
			prms[0].Value = orderedTestGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetQCResultsForOrderedTest.StoredProcName, prms);
			
			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/16/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3042"> 
		///		<ExpectedInput>The ID of OrderableTest</ExpectedInput>
		///		<ExpectedOutput>DataTable containing ReagentType associated with OrderableTest</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3043"> 
		///		<ExpectedInput>Invalid TestId</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the ReagentType associated with OrderableTest
		/// </summary>
		/// <param name="orderableTestId"></param>
		/// <returns></returns>
		public static DataTable GetReagentTypeForOrderableTest(int orderableTestId)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetReagentTypeForOrderableTest.OrderableTestId, System.Data.SqlDbType.Int)
			};
			prms[0].Value = orderableTestId;	

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetReagentTypeForOrderableTest.StoredProcName, prms);
			
			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/16/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3044"> 
		///		<ExpectedInput>The identifier of OrderedTest and the user name</ExpectedInput>
		///		<ExpectedOutput>True</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3045"> 
		///		<ExpectedInput>Invalid identifier, invalid user name</ExpectedInput>
		///		<ExpectedOutput>False</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Inactivates test results for ordered test
		/// </summary>
		/// <param name="dtOrderedTest"></param>
		/// <param name="dtInvalidatedTestException"></param>
		/// <returns></returns>
		public static bool InactivateTestResults(DataTable dtOrderedTest, DataTable dtInvalidatedTestException)
		{

			if ((int)dtOrderedTest.Rows[0][TABLE.OrderedTest.OrderableTestId] == -1)
			{
				dtOrderedTest.Columns.Remove(TABLE.OrderedTest.OrderableTestId);
			}

			DataTable[] dtArray = {dtOrderedTest, dtInvalidatedTestException};
			string[] spArray = {STOREDPROC.InactivateOrderedTestResults.StoredProcName,
									STOREDPROC.InsertExceptionPatientTestInactivated.StoredProcName};
			int result = (new StoredProcedure()).TransactionalGetValue(spArray, dtArray);

			return (result == 0);
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/16/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3046"> 
		///		<ExpectedInput>The valid OrderableTestId</ExpectedInput>
		///		<ExpectedOutput>The valid CprsOrderableItemId (interger > 0)</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3047"> 
		///		<ExpectedInput>Invalid OrderableTestId</ExpectedInput>
		///		<ExpectedOutput>-1</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the CprsOrderableitemID for OrderableTest
		/// </summary>
		/// <param name="orderableTestId"></param>
		/// <returns></returns>
		public static int GetCprsOrderable(int orderableTestId)
		{
			SqlParameter[] prms =
						{
							new SqlParameter(STOREDPROC.GetCprsOrderable.orderabletestid, System.Data.SqlDbType.Int),
			};
			prms[0].Value = orderableTestId;	

			DataSet ds = StoredProcedure.GetData(STOREDPROC.GetCprsOrderable.StoredProcName, prms);

			if (ds.Tables[0].Rows.Count == 0)
				return(-1);
			else if (ds.Tables[0].Rows[0].IsNull(TABLE.OrderableTest.CprsOrderableItemId) == true)
				return(-1);
			else
                return System.Convert.ToInt32(ds.Tables[0].Rows[0][TABLE.OrderableTest.CprsOrderableItemId]);
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/16/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3048"> 
		///		<ExpectedInput>The valid CprsOrderNumber</ExpectedInput>
		///		<ExpectedOutput>DataTable containing OrderedTest data associated with CprsOrderNumber</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3049"> 
		///		<ExpectedInput>Invalid CprsOrderNumber</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets the DataTable with all the data required by Vista
		/// </summary>
		/// <param name="cprsOrderNumber"></param>
		/// <returns></returns>
		public static DataTable GetOrderedTestForVista(string cprsOrderNumber)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetOrderDataTests.cprsordernumber, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.GetOrderDataTests.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = cprsOrderNumber;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetOrderDataTests.StoredProcName, prms);

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3090"> 
		///		<ExpectedInput>OrderedTest identifier</ExpectedInput>
		///		<ExpectedOutput>The DataTable containing historical test results</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3091"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the historical test results for the ordered test. This method returns the list of test results
		/// for the same type of testing and the same patient as the original test
		/// </summary>
		/// <param name="orderedTestGuid"></param>
		/// <returns></returns>
		public static DataTable GetHistoricalTestResultsForOrderedTest(Guid orderedTestGuid)
		{
			SqlParameter[] prms =
						{
							new SqlParameter(STOREDPROC.GetHistoricalResultsForOrderedTest.OrderedTestGuid, System.Data.SqlDbType.UniqueIdentifier)
						};
			prms[0].Value = orderedTestGuid;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetHistoricalResultsForOrderedTest.StoredProcName, prms);

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1305"> 
		///		<ExpectedInput>PatientSpecimenGuid</ExpectedInput>
		///		<ExpectedOutput>The DataTable containing ordered test results</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1315"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Get the ordered tests for a patient.
		/// </summary>
		/// <param name="patientSpecimenGuid"></param>
		/// <returns></returns>
		public static DataTable GetOrderedTests(Guid patientSpecimenGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetOrderedTests.patientspecimenguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetOrderedTests.divisioncode, System.Data.SqlDbType.Char)
			};
			prms[0].Value = patientSpecimenGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetOrderedTests.StoredProcName, prms);

			return ds.Tables[0];
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4167"> 
		///		<ExpectedInput>Valid Guid</ExpectedInput>
		///		<ExpectedOutput>DataSet</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4168"> 
		///		<ExpectedInput>EmptyGuid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetRecentTestOrders
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <returns></returns>
		public static DataSet GetRecentTestOrders(System.Guid patientGuid)
		{
			if (patientGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient unique identifier").ResString);
			}
			
			SqlParameter [] prms = 
			{
				new SqlParameter(STOREDPROC.GetRecentTestOrdersForPatient.patientguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetRecentTestOrdersForPatient.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = patientGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			return Common.StoredProcedure.GetData(STOREDPROC.GetRecentTestOrdersForPatient.StoredProcName, prms);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8191"> 
		///		<ExpectedInput>Valid patient guid</ExpectedInput>
		///		<ExpectedOutput>Count of the recent test orders</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8192"> 
		///		<ExpectedInput>Invalid patient Guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get a count of the recent test orders for a patient
		/// </summary>
		/// <param name="patientGuid">Patient Guid</param>
		/// <returns>Count of the recent test orders</returns>
		public static int GetRecentTestOrdersCount(System.Guid patientGuid)
		{
			if (patientGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient unique identifier").ResString);
			}
			
			int count = 0;

			SqlParameter [] prms = 
			{
				new SqlParameter(STOREDPROC.GetRecentTestOrdersForPatientCount.patientguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetRecentTestOrdersForPatientCount.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = patientGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetRecentTestOrdersForPatientCount.StoredProcName, prms);
			foreach(DataTable dt in ds.Tables)
			{
				foreach(DataRow dr in dt.Rows)
				{
					count += (int)dr["Total"];
				}
			}
			//
			return count;
		}



        ///<Developers>
        ///	<Developer>Doug Forst</Developer>
        ///</Developers>
        ///<SiteName></SiteName>
        ///<CreationDate>02/10/2016</CreationDate>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Get previous OrderedTest status
        /// </summary>
        /// <param name="orderedTestGuid">Patient Guid</param>
        /// <returns>Status of previous status</returns>
        public static TaskStatus GetOrderedTestToPreviousTaskStatus(System.Guid orderedTestGuid)
        {
            // Defect 269159 (for UC_115)
            if (orderedTestGuid == Guid.Empty)
            {
                throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("ordered test unique identifier").ResString);
            }

            TaskStatus taskStatus = TaskStatus.Unknown;

            SqlParameter[] prms = 
			{
				new SqlParameter(STOREDPROC.GetOrderedTestToPreviousTaskStatus.OrderedTestGuid, System.Data.SqlDbType.UniqueIdentifier)				
			};

            prms[0].Value = orderedTestGuid;

            var dt = Common.StoredProcedure.GetDataTable(STOREDPROC.GetOrderedTestToPreviousTaskStatus.StoredProcName, prms);

            if (dt != null &&
                dt.Rows.Count > 0)
            {
                var taskStatusCode = dt.Rows[0][TABLE.OrderedTest.TaskStatusCode];

                if (taskStatusCode != null)
                {
                    taskStatus = Utility.GetTaskStatusCodeFromString(taskStatusCode.ToString());
                }
            }

            return taskStatus;
        }

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4483"> 
		///		<ExpectedInput>Valid patient specimen unique identifier</ExpectedInput>
		///		<ExpectedOutput>Data table of ordered component and ordered test results</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4484"> 
		///		<ExpectedInput>Empty Guid</ExpectedInput>
		///		<ExpectedOutput>Argument exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static function to retrieve a list of completed tests for a given specimen
		/// </summary>
		/// <param name="patientSpecimenGuid">Patient specimen unique identifier</param>
		/// <returns>Data table of specimen test information</returns>
		public static DataTable GetTestOrders(Guid patientSpecimenGuid)
		{
			if (patientSpecimenGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient specimen unique identifier").ResString);
			}
			
			SqlParameter [] prms = 
			{
				new SqlParameter(STOREDPROC.GetTestOrders.patientspecimenguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetTestOrders.divisioncode, System.Data.SqlDbType.Char)
			};
			prms[0].Value = patientSpecimenGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			DataTable dt = Common.StoredProcedure.GetData(STOREDPROC.GetTestOrders.StoredProcName, prms).Tables[0];
			return dt;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4479"> 
		///		<ExpectedInput>Valid ordered test unique identifier</ExpectedInput>
		///		<ExpectedOutput>Data table of specimen test results</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4480"> 
		///		<ExpectedInput>Empty Guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static function to retrieve a list of specimen test results for a given ordered test
		/// </summary>
		/// <param name="orderedTestGuid">Ordered test unique identifier</param>
		/// <returns>Data table of specimen test information</returns>
		public static DataTable GetSpecimenTestResultsForOrderedTest(Guid orderedTestGuid)
		{
			if (orderedTestGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("ordered test unique identifier").ResString);
			}
			//
			SqlParameter [] prms = 
			{
				new SqlParameter(STOREDPROC.GetSpecimenTestResultsForOrderedTest.orderedtestguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetSpecimenTestResultsForOrderedTest.divisioncode, System.Data.SqlDbType.Char)
			};
			prms[0].Value = orderedTestGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			DataTable dt = Common.StoredProcedure.GetData(STOREDPROC.GetSpecimenTestResultsForOrderedTest.StoredProcName, prms).Tables[0];
			return dt;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4481"> 
		///		<ExpectedInput>Valid ordered test unique identifier and reportable test id</ExpectedInput>
		///		<ExpectedOutput>Data table of subset of entire TAS specimen test results</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4482"> 
		///		<ExpectedInput>Empty Guid</ExpectedInput>
		///		<ExpectedOutput>Argument exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static function to retrieve a list of specimen test results for a given ordered test within a TAS grouping
		/// </summary>
		/// <param name="orderedTestGuid">Ordered test unique identifier</param>
		/// <param name="reportableTestID">ID of sub-test</param>
		/// <returns>Data table of specimen test information</returns>
		public static DataTable GetSpecimenTestResultsForOrderedTest(Guid orderedTestGuid, int reportableTestID)
		{
			if (orderedTestGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("ordered test unique identifier").ResString);
			}
			//
			SqlParameter [] prms = 
			{
				new SqlParameter(STOREDPROC.GetSpecimenTestResultsForOrderedTest.orderedtestguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetSpecimenTestResultsForOrderedTest.reportabletestid, System.Data.SqlDbType.Int),
				new SqlParameter(STOREDPROC.GetSpecimenTestResultsForOrderedTest.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = orderedTestGuid;
			prms[1].Value = reportableTestID;
			prms[2].Value = Common.LogonUser.LogonUserDivisionCode;

			DataTable dt = Common.StoredProcedure.GetData(STOREDPROC.GetSpecimenTestResultsForOrderedTest.StoredProcName, prms).Tables[0];
			return dt;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4473"> 
		///		<ExpectedInput>Valid ordered test unique identifier</ExpectedInput>
		///		<ExpectedOutput>Data table of completed blood unit test results</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4474"> 
		///		<ExpectedInput>Invalid ordered test unique identifier</ExpectedInput>
		///		<ExpectedOutput>Argument exception thrown</ExpectedOutput>>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static function to retrieve a list of blood unit test results for a given ordered test (XMs only)
		/// </summary>
		/// <param name="orderedTestGuid">Ordered test unique identifier</param>
		/// <returns>Data table of blood unit test information</returns>
		public static DataTable GetBloodUnitTestResultsForOrderedTest(Guid orderedTestGuid)
		{
			if (orderedTestGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("ordered test unique identifier").ResString);
			}
			
			SqlParameter [] prms = 
			{
				new SqlParameter(STOREDPROC.GetBloodUnitTestResultsForOrderedTest.orderedtestguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetBloodUnitTestResultsForOrderedTest.divisioncode, System.Data.SqlDbType.Char)
			};
			prms[0].Value = orderedTestGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			DataTable dt = Common.StoredProcedure.GetData(STOREDPROC.GetBloodUnitTestResultsForOrderedTest.StoredProcName, prms).Tables[0];
			return dt;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4465"> 
		///		<ExpectedInput>Valid ordered test and blood unit unique identifiers</ExpectedInput>
		///		<ExpectedOutput>Data table of completed blood unit test results</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4466"> 
		///		<ExpectedInput>Empty ordered test and blood unit unique identifiers</ExpectedInput>
		///		<ExpectedOutput>Argumentexception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static function to retrieve a list of blood unit test results for a given ordered test (XMs only)
		/// </summary>
		/// <param name="orderedTestGuid">Ordered test unique identifier</param>
		/// <param name="bloodUnitGuid">Blood unit unique identifier</param>
		/// <returns>Data table of blood unit test information</returns>
		public static DataTable GetBloodUnitTestResultsForOrderedTest(Guid orderedTestGuid, Guid bloodUnitGuid)
		{
			if (orderedTestGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("ordered test unique identifier").ResString);
			}
			
			if (bloodUnitGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("blood unit unique identifier").ResString);
			}
		
			SqlParameter [] prms = 
			{
				new SqlParameter(STOREDPROC.GetBloodUnitTestResultsForOrderedTest.orderedtestguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetBloodUnitTestResultsForOrderedTest.bloodunitguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetBloodUnitTestResultsForOrderedTest.divisioncode, System.Data.SqlDbType.Char)
			};
			prms[0].Value = orderedTestGuid;
			prms[1].Value = bloodUnitGuid;
			prms[2].Value = Common.LogonUser.LogonUserDivisionCode;

			DataTable dt = Common.StoredProcedure.GetData(STOREDPROC.GetBloodUnitTestResultsForOrderedTest.StoredProcName, prms).Tables[0];
			return dt;
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/3/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4277"> 
		///		<ExpectedInput>DivisionCode, Hours Needed, Date LastChecked</ExpectedInput>
		///		<ExpectedOutput>DataTable of order information</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4278"> 
		///		<ExpectedInput>Null or empty division code, hours not 0</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetRequiredOrders
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <param name="hoursNeeded"></param>
		/// <param name="lastChecked"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetRequiredOrders(string divisionCode, int hoursNeeded, System.DateTime lastChecked)
		{	
			if (divisionCode == null || divisionCode.Equals(string.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("divisionCode").ResString);
			}
			
			if (hoursNeeded < 0)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("hoursNeeded").ResString);
			}
			
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetNewOrderedTestsForRequiredTimeFrame.divisioncode, System.Data.SqlDbType.Char),
				new SqlParameter(STOREDPROC.GetNewOrderedTestsForRequiredTimeFrame.lastchecked, System.Data.SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.GetNewOrderedTestsForRequiredTimeFrame.requiredhours, System.Data.SqlDbType.Int)
			};
			//
			prms[0].Value = divisionCode;
			prms[1].Value = lastChecked;
			prms[2].Value = hoursNeeded;
			//
			return Common.StoredProcedure.GetData(STOREDPROC.GetNewOrderedTestsForRequiredTimeFrame.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/3/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4279"> 
		///		<ExpectedInput>DivisionCode, DateLastChecked</ExpectedInput>
		///		<ExpectedOutput>DataTable of Order Information</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4280"> 
		///		<ExpectedInput>Null or Empty division code</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetOrdersWithSpecimenInBloodBank
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <param name="lastChecked"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetOrdersWithSpecimenInBloodBank(string divisionCode, System.DateTime lastChecked)
		{	
			if (divisionCode == null || divisionCode.Equals(string.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("divisionCode").ResString);
			}

			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetNewOrderedTestsWithNoRequiredSpecimenOrSpecimenOnFile.divisioncode, System.Data.SqlDbType.Char),
				new SqlParameter(STOREDPROC.GetNewOrderedTestsWithNoRequiredSpecimenOrSpecimenOnFile.lastchecked, System.Data.SqlDbType.DateTime)
			};
			//
			prms[0].Value = divisionCode;
			prms[1].Value = lastChecked;
			//
			return Common.StoredProcedure.GetData(STOREDPROC.GetNewOrderedTestsWithNoRequiredSpecimenOrSpecimenOnFile.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/3/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4281"> 
		///		<ExpectedInput>DivisonCode, Datelastchecked</ExpectedInput>
		///		<ExpectedOutput>DataTable of Order Information</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4282"> 
		///		<ExpectedInput>Null or Empty DivisionCode</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetNewOrders
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <param name="lastChecked"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetNewOrders(string divisionCode, System.DateTime lastChecked)
		{	
			if (divisionCode == null || divisionCode.Equals(string.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("divisionCode").ResString);
			}

			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetNewOrderedTests.divisioncode, System.Data.SqlDbType.Char),
				new SqlParameter(STOREDPROC.GetNewOrderedTests.lastchecked, System.Data.SqlDbType.DateTime)
			};
			//
			prms[0].Value = divisionCode;
			prms[1].Value = lastChecked;
			//
			return Common.StoredProcedure.GetData(STOREDPROC.GetNewOrderedTests.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8051"> 
		///		<ExpectedInput>Valid OrderedTestGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8052"> 
		///		<ExpectedInput>Invalid OrderedTestGuid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the count of issued units for ordered test
		/// </summary>
		/// <param name="orderedTestGuid"></param>
		/// <returns></returns>
		public static DataTable GetEmergencyIssuedUnitsForOrderedTest(Guid orderedTestGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetEmergencyIssuedUnitsCountForOrderedTest.orderedtestguid, System.Data.SqlDbType.UniqueIdentifier)
			};

			prms[0].Value = orderedTestGuid;
			
			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetEmergencyIssuedUnitsCountForOrderedTest.StoredProcName, prms);

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/7/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8166"> 
		///		<ExpectedInput>Valid ordered test guid</ExpectedInput>
		///		<ExpectedOutput>Antisera Ordered Test table with 4 rows returned</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8167"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty data table</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get Antisera Ordered Test records associated with an ordered test
		/// </summary>
		/// <param name="orderedTestGuid">Ordered Test Guid</param>
		/// <returns>Antisera Ordered Test table</returns>
		public static DataTable GetAntiseraOrderedTests(Guid orderedTestGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetAntiseraOrderedTests.orderedtestguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetAntiseraOrderedTests.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = orderedTestGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;
			
			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetAntiseraOrderedTests.StoredProcName, prms);

			return ds.Tables[0];
		}
	}
}
